﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins.Provider
{
    public class ProviderServiceCreate : IPlugin
    {
        private const string PluginName = "ProviderServiceCreate";
        private const string ProcessDateAttribute = "ppms_validationcompletedate";
        private IOrganizationService _service;
        private ITracingService _tracingService;

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_providerservice")
                {
                    _tracingService.Trace("Entity type does not match! [{0}]", entity.LogicalName);
                    return;
                }

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Verify service is pending
                    var statusCode = entity.GetAttributeValue<OptionSetValue>("statuscode");
                    if (statusCode != null && statusCode.Value != (int)ppms_providerservice_StatusCode.Pending)
                    {
                        _tracingService.Trace("Provider service not in pending status.");
                        return;
                    }

                    // Get related entities to activate
                    _tracingService.Trace("Retrieve Provider and related entities");
                    var providerService = GetProviderService(_service, entity.Id);
                    if (providerService == null)
                    {
                        _tracingService.Trace("Failed to retrieve provider service and related entities");
                        return;
                    }
                    _tracingService.Trace("Provider Service Retrieved");

                    IList<SetStateRequest> requests = new List<SetStateRequest>();
                    bool isProviderActive = false;
                    bool isAgreementActive = false;

                    // Get associated provider to determine if in Active status
                    Account provider = providerService.ppms_account_ppms_providerservice;
                    if (provider == null)
                    {
                        _tracingService.Trace("Associated provider does not exist.");
                        return;
                    }
                    else
                    {
                        isProviderActive = PpmsHelper.DoesStateMatch(provider, (int)AccountState.Active);
                    }

                    // Get associated agreement to determine if in Active status
                    ppms_provideragreement agreement = providerService.ppms_provideragreement_ppms_providerservice_ProviderAgreement;
                    // Only sync status if provider service has associated agreement
                    if (agreement != null)
                    {
                        isAgreementActive = PpmsHelper.DoesStateMatch(agreement, (int)ppms_provideragreementState.Active);
                        var isAgreementActiveActive = PpmsHelper.DoesStatusMatch(agreement, (int)ppms_provideragreementState.Active, (int)ppms_provideragreement_StatusCode.Active);

                        if (isProviderActive && isAgreementActive)
                        {
                            if (isAgreementActiveActive)
                            {
                                _tracingService.Trace("Set provider service to ACTIVE.");
                                requests.Add(PpmsHelper.GetActivateRequest(providerService, (int)ppms_providerservice_StatusCode.Active));
                            }
                            else
                            {
                                _tracingService.Trace("Set provider service to PENDING.");
                                requests.Add(PpmsHelper.GetActivateRequest(providerService, (int)ppms_providerservice_StatusCode.Pending));
                            }
                        }
                        else
                        {
                            _tracingService.Trace("Set provider service to INACTIVE.");
                            requests.Add(PpmsHelper.GetDeactivateRequest(providerService, (int)ppms_providerservice_StatusCode.Inactive));
                        }
                    }
                    else _tracingService.Trace("Provider service does not have associated agreement.");

                    // Execute requests
                    if (requests.Count > 0)
                    {
                        _tracingService.Trace("Execute requests {0}", requests.Count);
                        foreach (var request in requests)
                        {
                            _service.Execute(request);
                        }
                    }
                    _tracingService.Trace("Execution complete.");
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        private ppms_providerservice GetProviderService(IOrganizationService service, Guid id)
        {
            using (var context = new PpmsContext(service))
            {
                var entity = context.ppms_providerserviceSet.FirstOrDefault(a => a.ppms_providerserviceId == id);
                if (entity != null)
                {
                    context.LoadProperty(entity, new Relationship("ppms_account_ppms_providerservice"));
                    context.LoadProperty(entity, new Relationship("ppms_provideragreement_ppms_providerservice_ProviderAgreement"));
                }
                return entity;
            }
        }

        private Entity GetBatchDetail(Guid batchDetailId)
        {
            return _service.Retrieve("ppms_batchdetail", batchDetailId, new ColumnSet(new string[] { "ppms_batchdetailid", "ppms_name", "ppms_batch", ProcessDateAttribute }));
        }

        private Account GetProviderWithBatchDetail(Guid providerId)
        {
            using (var context = new PpmsContext(_service))
            {
                var provider = context.AccountSet.FirstOrDefault(a => a.AccountId == providerId);
                if (provider != null)
                {
                    context.LoadProperty(provider, new Relationship("ppms_account_batchdetail_provider"));
                }
                return provider;
            }
        }

        private Entity GetCurrentBatchDetail(EntityReference providerRef)
        {
            if (providerRef != null)
            {
                var provider = GetProviderWithBatchDetail(providerRef.Id);
                if (provider != null)
                {
                    // determine target batch detail
                    var children = provider.ppms_account_batchdetail_provider;
                    if (children != null && children.Any())
                    {
                        return provider.ppms_account_batchdetail_provider.OrderBy(o => o.CreatedOn).Where(a => a.ppms_validationcompletedate == null).FirstOrDefault();
                    }
                }
            }

            return null;
        }

        private IEnumerable<Entity> GetCurrentBatchDetails(EntityReference providerRef)
        {
            if (providerRef != null)
            {
                var provider = GetProviderWithBatchDetail(providerRef.Id);
                if (provider != null)
                {
                    // determine target batch detail
                    var children = provider.ppms_account_batchdetail_provider;
                    if (children != null && children.Any())
                    {
                        return provider.ppms_account_batchdetail_provider.OrderBy(o => o.CreatedOn).Where(a => a.ppms_validationcompletedate == null);
                    }
                }
            }

            return null;
        }
    }
}
